/*  Prüfung auf Unregelmäßigkeiten in der Abarbeitung bzw. bestehenden Terminierung von einen gegebenen AG.
*/
SELECT tsystem.function__drop_by_regex('resource_timeline__ab2_detect_issues', 'scheduling', _commit => true );
CREATE OR REPLACE FUNCTION scheduling.resource_timeline__ab2_detect_issues(
    _ab2_id int,
    _window tsrange = tsrange( now()::timestamp, 'infinity'::timestamp ),
    _skip_block_validation bool = false
) RETURNS jsonb AS $$
DECLARE
    _blocks_record  record;
    _ab2_record record;

    _last_blocktime_end timestamp;

    _work_required  numeric;
    _work_done      numeric;

    _debug_prefix     text := '[scheduling.resource_timeline__ab2_detect_issues » ab2: ' || _ab2_id || ']';

    _result jsonb :=
       '{
          "has_issues": false,
          "issues": {
              "ab2": [],
              "resource_timeline": []
          }
        }';
BEGIN
    -- Daten des AG holen.
    SELECT a2_id, a2_ab_ix, a2_n, min(ti_date_start) as min_ti_date_start, max(ti_date_end) as max_ti_date_end, ti_resource_id
    INTO _ab2_record
    FROM ab2
    LEFT JOIN scheduling.resource_timeline ON a2_id = ti_a2_id
    WHERE a2_id = _ab2_id
    GROUP BY a2_id, a2_ab_ix, a2_n, ti_resource_id;

    -- Prüfung der logischen Integrität der Task-Daten in der Tabelle "resource_timeline".
    -- Z.B. wird geprüft, ist die Belegung größer 1 oder liegt Abarbeitung außerhalb der Arbeitszeiten.
    -- Diese Integritätsprüfung kann mit dem Parameter "_skip_block_validation" übersprungen werden.
    -- TODO AXS: Sind diese Integritätsprüfungen wirklich standardmäßig notwendig? Wenn diese anschlagen, dann haben wir im Algorithmus etwas falsch gemacht. Wir sollten nach der Testphase aus Performancegründen den Standardwert für den Parameter "_skip_block_validation" auf true stellen.
    IF ( NOT _skip_block_validation ) THEN

        FOR _blocks_record IN
            SELECT ti_id, scheduling.resource_timeline__validate_block( resource_timeline ) result
              FROM scheduling.resource_timeline
             WHERE ti_a2_id = _ab2_id
             ORDER BY ti_date_start
        LOOP

            IF ( _blocks_record.result->'has_issues' = 'true' ) THEN
                _result := jsonb_set( _result, '{ has_issues }', 'true' );
                _result := jsonb_insert(
                    _result,
                    '{ issues, resource_timeline, 0 }',
                    format(
                        '{ "ti_id": %s, "error": %s }',
                        _blocks_record.ti_id,
                        _blocks_record.result
                    )::jsonb
                );
            END IF;

        END LOOP;

    END IF;

    -- Prüfung der Arbeitszeiten.
    -- TODO AXS: Warum Prüfung bis Tagesanfang? Möglichkeit einer fehlerhaften Alamierung?
    _work_required := scheduling.resource_timeline__ab2__get_scheduled_time( _ab2_id => _ab2_id, _timeframe_end => date_trunc( 'day' , lower( _window ) ) );
    _work_required := coalesce( _work_required, 0 );
    _work_done := scheduling.ab2__worktime_done__get( _ab2_id, date_trunc( 'day' , lower( _window ) ) );

    -- Es wurde weniger Arbeit erledigt, als geplant. Mähhh, wir haben Überhang:-(
    IF ( _work_done < _work_required ) THEN

        _result := jsonb_set( _result, '{ has_issues }', 'true' );
        _result := jsonb_insert(
            _result,
            '{ issues, ab2, 0 }',
            format('"amount work accomplished %s is lower than required %s"', _work_done::int, _work_required::int )::text::jsonb
        );
    END IF;

    -- Es wurde mehr Arbeit erledigt, als geplant. Juhu, wir sind im Vorsprung:-)
    IF ( _work_done > _work_required ) THEN
        _result := jsonb_insert(
            _result,
            '{ issues, ab2, 0 }',
            format('"amount work accomplished %s is higher than required %s"', _work_done::int, _work_required::int )::text::jsonb
        );
    END IF;

    -- Ermittle das Ende der Blocktime-Einträge des aktuellen und vorhergehenden AG.
    SELECT max( ti_date_end )
    INTO _last_blocktime_end
    FROM scheduling.resource_timeline
    WHERE
              ti_type = 'task.blocktime'
          AND ti_a2_id IN (
                            SELECT a2_id
                            FROM ab2
                            WHERE
                                      a2_ab_ix = _ab2_record.a2_ab_ix
                                  AND a2_n <= _ab2_record.a2_n
                          )
    ;
    -- Die Blockierungen ragen in einen nachfolgenden AG.
    IF ( _last_blocktime_end > _ab2_record.min_ti_date_start ) THEN
        _result := jsonb_set( _result, '{has_issues}', 'true' );
        _result := jsonb_insert(
            _result,
            '{ issues, ab2, 0 }',
            '"end of last task.blocktime after start of next task"'::jsonb
        );
    END IF;

    RETURN _result;

END $$ language plpgsql STABLE STRICT;